<?php
/**
 * Plugin Name: SCR Layer Exploder (Capas con animación)
 * Description: Muestra una imagen “apilada” que, al hacer clic, se separa por capas con animación (ideal para mostrar 7 capas con descripciones). Usa shortcode: [scr_layer_exploder]
 * Version: 1.0.9
 * Author: SCR Business
 * License: GPLv2 or later
 * Text Domain: scr-layer-exploder
 */

if (!defined('ABSPATH')) exit;

class SCR_Layer_Exploder {
  const VERSION = '1.0.9';
  const HANDLE  = 'scr-layer-exploder';

  public function __construct() {
    add_action('wp_enqueue_scripts', [$this, 'register_assets']);
    add_shortcode('scr_layer_exploder', [$this, 'shortcode']);
  }

  public function register_assets() {
    $url = plugin_dir_url(__FILE__);
    wp_register_style(self::HANDLE, $url . 'assets/css/scr-layer-exploder.css', [], self::VERSION);
    wp_register_script(self::HANDLE, $url . 'assets/js/scr-layer-exploder.js', [], self::VERSION, true);
  }

  private function sanitize_layers_json($json_raw) {
    if (!$json_raw) return [];

    // Elementor/WordPress may add slashes and HTML entities; normalize.
    $json_raw = wp_unslash($json_raw);
    $json_raw = html_entity_decode($json_raw, ENT_QUOTES, 'UTF-8');
    $json_raw = trim($json_raw);

    // If URL-encoded JSON was provided, decode it.
    if (strpos($json_raw, '%5B') !== false || strpos($json_raw, '%7B') !== false) {
      $maybe = urldecode($json_raw);
      if ($maybe) $json_raw = $maybe;
    }

    // If base64 JSON was provided (layers_b64 use-case), decode it.
    if (preg_match('/^[A-Za-z0-9+\/=
]+$/', $json_raw) && strlen($json_raw) > 40 && strpos($json_raw, '[') === false) {
      $decoded = base64_decode($json_raw, true);
      if ($decoded) $json_raw = $decoded;
    }

    // Remove newlines that can break json_decode in some cases
    $json_raw = preg_replace("/
|
|
/", "", $json_raw);

    $layers = json_decode($json_raw, true);
    if (!is_array($layers)) return [];

    $clean = [];
    foreach ($layers as $layer) {
      if (!is_array($layer)) continue;
      $clean[] = [
        'img'   => isset($layer['img']) ? esc_url_raw($layer['img']) : '',
        'title' => isset($layer['title']) ? sanitize_text_field($layer['title']) : '',
        'text'  => isset($layer['text']) ? wp_kses_post($layer['text']) : '',
        'side'  => isset($layer['side']) ? sanitize_text_field($layer['side']) : '',
        'y'     => isset($layer['y']) ? floatval($layer['y']) : 0,
      ];
    }
    return $clean;
  }

  private function infer_layer_images($base_url, $layers, $prefix='capa-', $ext='png') {
    if (empty($base_url) || empty($layers)) return $layers;

    // Only if all imgs are empty
    $all_empty = true;
    foreach ($layers as $l) { if (!empty($l['img'])) { $all_empty = false; break; } }
    if (!$all_empty) return $layers;

    $parts = wp_parse_url($base_url);
    if (empty($parts['scheme']) || empty($parts['host']) || empty($parts['path'])) return $layers;

    $path = $parts['path'];
    $dir  = preg_replace('~[^/]+$~', '', $path); // directory of base
    $base_dir_url = $parts['scheme'].'://'.$parts['host'].$dir;

    // handle wp_parse_url missing port
    if (!empty($parts['port'])) {
      $base_dir_url = $parts['scheme'].'://'.$parts['host'].':'.$parts['port'].$dir;
    }

    // We assume 7 layers by default; map idx 0..6 -> capa-7..1 (top to bottom)
    $n = count($layers);
    for ($i=0; $i<$n; $i++){
      $num = max(1, min(7, 7 - $i));
      $guess = $base_dir_url . $prefix . $num . '.' . $ext;
      $layers[$i]['img'] = esc_url_raw($guess);
    }
    return $layers;
  }

  // Igual que infer_layer_images(), pero SIN validar existencia (HEAD/GET),
  // porque algunos hosts bloquean HEAD o devuelven 403 y eso deja img vacío.
  private function infer_layer_images_blind($base_url, $layers, $prefix='capa-', $ext='.png') {
    if (empty($base_url) || empty($layers) || !is_array($layers)) { return $layers; }

    $parts = wp_parse_url($base_url);
    if (empty($parts['scheme']) || empty($parts['host']) || empty($parts['path'])) { return $layers; }

    $dir = rtrim(dirname($parts['path']), '/');
    $base_dir = $parts['scheme'].'://'.$parts['host'].$dir.'/';

    $total = count($layers);
    for ($i=0; $i<$total; $i++) {
      if (!isset($layers[$i]['img']) || $layers[$i]['img']==='') {
        $num = $total - $i; // idx 0 => capa-7
        $layers[$i]['img'] = esc_url_raw($base_dir . $prefix . $num . $ext);
      }
    }
    return $layers;
  }


  private function build_layers_from_flat_atts($atts) {
    $layers = [];
    for ($i=1; $i<=7; $i++) {
      $img = isset($atts["img{$i}"]) ? trim($atts["img{$i}"]) : '';
      $title = isset($atts["title{$i}"]) ? trim($atts["title{$i}"]) : '';
      $text = isset($atts["text{$i}"]) ? $atts["text{$i}"] : '';
      $side = isset($atts["side{$i}"]) ? trim($atts["side{$i}"]) : '';
      $y = isset($atts["y{$i}"]) ? floatval($atts["y{$i}"]) : 0;

      if ($img === '' && $title === '' && trim(strip_tags($text)) === '') continue;

      $layers[] = [
        'img' => esc_url_raw($img),
        'title' => sanitize_text_field($title),
        'text' => wp_kses_post($text),
        'side' => sanitize_text_field($side),
        'y' => $y,
      ];
    }
    return $layers;
  }

  private function get_layers_from_atts_or_content($atts_layers, $content) {
    $layers = $this->sanitize_layers_json($atts_layers);
    if (!empty($layers)) return $layers;

    if (!empty($content)) {
      $layers2 = $this->sanitize_layers_json($content);
      if (!empty($layers2)) return $layers2;
    }
    return [];
  }

  public function shortcode($atts = [], $content = null) {
    $atts = shortcode_atts([
      'base'       => '',      // URL imagen principal (producto “cerrado”)
      'layers'     => '',      // JSON array [{img,title,text}, ...] (7 items recomendado)
      'gap'        => '34',    // separación vertical en px entre capas
      'start_open' => '0',     // 1=arranca abierto
      'width'      => '720',   // ancho máximo en px (responsive)
      'align'      => 'center', // left|center|right
      'show_labels'=> '1',      // 1=mostrar labels laterales
      'infer_layers'=>'1',      // 1=si no llegan imgs, inferir por nombre desde base
      'infer_prefix'=>'capa-',  // prefijo de archivos capa-1..7
      'infer_ext'=>'png'        // extensión
    ], $atts, 'scr_layer_exploder');

    $base = esc_url($atts['base']);
    $raw_layers = !empty($atts['layers_b64']) ? $atts['layers_b64'] : $atts['layers'];
    $layers = $this->get_layers_from_atts_or_content($raw_layers, $content);
    if (empty($layers)) {
      $layers = $this->build_layers_from_flat_atts($atts);
    }

    // Inferir URLs de imágenes si las capas llegaron sin 'img' (muy común con Elementor)
    if (!empty($layers) && !empty($base)) {
      $needs_infer = true;
      foreach ($layers as $l) { if (!empty($l['img'])) { $needs_infer = false; break; } }
      if ($needs_infer) {
        $layers = $this->infer_layer_images($base, $layers, $atts['infer_prefix'], $atts['infer_ext']);
      // Si sigue vacío por bloqueos de HEAD/403, rellenar a ciegas
      $layers = $this->infer_layer_images_blind($base, $layers, $atts['infer_prefix'], $atts['infer_ext']);
      }
    }

    // Fallback demo (si no pasan layers)
    if (empty($layers)) {
      $layers = [
        ['img'=>'', 'title'=>'Capa Superior', 'text'=>'En algodón suave delicada y de rápida absorción.'],
        ['img'=>'', 'title'=>'Cinta de Anión (Iones Negativos)', 'text'=>'Emite partículas de oxígeno puro.'],
        ['img'=>'', 'title'=>'Gel Súper Absorbente', 'text'=>'Congela rápidamente el líquido evitando el regreso.'],
        ['img'=>'', 'title'=>'Hoja de Papel', 'text'=>'Esterilizado libre de polvo, usado para envolver el agente absorbente.'],
        ['img'=>'', 'title'=>'Doble Hoja de Papel', 'text'=>'Esterilizado, libre de polvo, usado para envolver el gel absorbente.'],
        ['img'=>'', 'title'=>'Capa de Algodón', 'text'=>'Envuelve el material absorbente previniendo que te manches.'],
        ['img'=>'', 'title'=>'Capa Posterior', 'text'=>'Membrana respirable, permeable al aire pero no al agua.'],
      ];
    }

    // Ensure at least 1 layer
    if (empty($layers)) return '';

    // Enqueue assets only when shortcode used
    wp_enqueue_style(self::HANDLE);
    wp_enqueue_script(self::HANDLE);

    $uid = 'scrle_' . wp_generate_uuid4();

    $data = [
      'base'       => $base,
      'layers'     => $layers,
      'gap'        => (int)$atts['gap'],
      'startOpen'  => ($atts['start_open'] === '1' || strtolower($atts['start_open']) === 'true'),
      'maxWidth'   => (int)$atts['width'],
      'align'      => in_array($atts['align'], ['left','center','right'], true) ? $atts['align'] : 'center',
      'showLabels'=> ($atts['show_labels'] === '1' || strtolower($atts['show_labels']) === 'true'),
    ];

    // Pass config to JS via data attribute (safer with multiple instances)
    $json = wp_json_encode($data);

    ob_start(); ?>
      <div class="scrle-wrap scrle-align-<?php echo esc_attr($data['align']); ?> <?php echo !empty($data['showLabels']) ? 'scrle-labels-on' : 'scrle-labels-off'; ?>" style="--scrle-maxw: <?php echo (int)$data['maxWidth']; ?>px" data-scrle="<?php echo esc_attr($json); ?>" data-scrle-debug="<?php
            $first = isset($layers[0]['img']) ? $layers[0]['img'] : '';
            $parts = wp_parse_url($base);
            $dir = (!empty($parts['path'])) ? preg_replace('~[^/]+$~', '', $parts['path']) : '';
            $baseDir = (!empty($parts['scheme']) && !empty($parts['host']) && $dir) ? ($parts['scheme'].'://'.$parts['host'].$dir) : '';
            echo esc_attr('layers='.count($layers).';first='.$first.';baseDir='.$baseDir);
          ?>" id="<?php echo esc_attr($uid); ?>">
        <div class="scrle-stage" aria-label="Explosión por capas" role="group">
          <button class="scrle-hit" type="button" aria-expanded="false" aria-controls="<?php echo esc_attr($uid); ?>_panel">
            <span class="scrle-sr">Ver capas</span>
          </button>

          <div class="scrle-stack" aria-hidden="true">
            <?php foreach ($layers as $i => $layer): ?>
              <div class="scrle-layer" data-idx="<?php echo (int)$i; ?>">
                <?php if (!empty($layer['img'])): ?>
                  <img class="scrle-layer-img" src="<?php echo esc_url($layer['img']); ?>" alt="<?php echo esc_attr($layer['title']); ?>">
                <?php else: ?>
                  <!-- Si no hay imagen por capa, se usa una “lámina” generada por CSS -->
                  <div class="scrle-layer-fallback" aria-hidden="true"></div>
                <?php endif; ?>
              </div>
            <?php endforeach; ?>
          </div>

          <div class="scrle-callouts" aria-hidden="true">
            <?php foreach ($layers as $i => $layer):
              $side = !empty($layer['side']) ? $layer['side'] : (($i % 2 === 0) ? 'left' : 'right');
              $y = isset($layer['y']) && $layer['y'] > 0 ? $layer['y'] : (12 + ($i * (76 / max(1, count($layers)-1))));
              $y = max(5, min(95, $y));
            ?>
              <div class="scrle-callout scrle-<?php echo esc_attr($side); ?>" data-idx="<?php echo (int)$i; ?>" style="top: <?php echo esc_attr($y); ?>%;">
                <div class="scrle-callout-title"><?php echo esc_html($layer['title']); ?></div>
                <div class="scrle-callout-text"><?php echo wp_kses_post($layer['text']); ?></div>
              </div>
            <?php endforeach; ?>
          </div>

          <?php if (!empty($base)): ?>
            <img class="scrle-base" src="<?php echo esc_url($base); ?>" alt="Producto" loading="lazy">
          <?php endif; ?>
        </div>

        <div class="scrle-panel" id="<?php echo esc_attr($uid); ?>_panel">
          <div class="scrle-panel-head">
            <div class="scrle-title">Capas</div>
            <button type="button" class="scrle-reset">Reiniciar</button>
          </div>

          <ol class="scrle-list">
            <?php foreach ($layers as $i => $layer): ?>
              <li class="scrle-item" data-idx="<?php echo (int)$i; ?>">
                <div class="scrle-item-title"><?php echo esc_html($layer['title']); ?></div>
                <div class="scrle-item-text"><?php echo wp_kses_post($layer['text']); ?></div>
              </li>
            <?php endforeach; ?>
          </ol>

          <div class="scrle-hint">Tip: haz clic en la toalla (o botón) para abrir/cerrar. Pasa el mouse sobre una capa para resaltarla.</div>
        </div>
      </div>
    <?php
    return ob_get_clean();
  }
}

new SCR_Layer_Exploder();
