(function(){
  function clamp(n, a, b){ return Math.max(a, Math.min(b, n)); }

  function initExploder(root){
    if (!root || root.__scrle_init) return;
    root.__scrle_init = true;

    let cfg = {};
    try { cfg = JSON.parse(root.getAttribute('data-scrle') || '{}'); } catch(e){ cfg = {}; }

    const gap = Number(cfg.gap || 34);
    const startOpen = !!cfg.startOpen;

    const stage   = root.querySelector('.scrle-stage');
    const hit     = root.querySelector('.scrle-hit');
    const layers  = Array.from(root.querySelectorAll('.scrle-layer'));
    const items   = Array.from(root.querySelectorAll('.scrle-item'));
    const callouts= Array.from(root.querySelectorAll('.scrle-callout'));
    const resetBtn= root.querySelector('.scrle-reset');

    if (!stage || layers.length === 0) return;

    let open = false;

    // Ensure top layer is visually on top
    layers.forEach((el, idx) => {
      el.style.zIndex = String(20 + (layers.length - idx));
    });

    function setActive(idx){
      items.forEach(it => it.classList.toggle('is-active', Number(it.dataset.idx) === idx));
      callouts.forEach(co => co.classList.toggle('is-active', Number(co.dataset.idx) === idx));
    }
    function clearActive(){
      items.forEach(it => it.classList.remove('is-active'));
      callouts.forEach(co => co.classList.remove('is-active'));
    }

    function applyTransforms(){
      layers.forEach((el, i) => {
        const k = i;
        const lift = -gap * k;
        const x = (k % 2 === 0 ? -1 : 1) * clamp(6 + k*1.2, 6, 18);
        const rot = (k % 2 === 0 ? -1 : 1) * clamp(2 + k*0.35, 2, 6);
        const z = k * 2;

        if (open){
          el.classList.add('is-open');
          el.style.transform = `translate3d(-50%, -50%, ${z}px) translate3d(${x}px, ${lift}px, 0) rotateX(10deg) rotateZ(${rot}deg)`;
        } else {
          el.classList.remove('is-open');
          el.style.transform = `translate3d(-50%, -50%, 0) scale(0.98)`;
        }
      });

      if (hit) hit.setAttribute('aria-expanded', open ? 'true' : 'false');
      root.classList.toggle('is-open', open);
      if (!open) clearActive();
    }

    function toggle(){
      open = !open;
      applyTransforms();
    }
    function reset(){
      open = false;
      applyTransforms();
    }

    // Primary: click on full stage (but ignore clicks in panel + reset)
    stage.addEventListener('click', (e) => {
      if (e.target && e.target.closest('.scrle-panel')) return;
      if (e.target && e.target.closest('.scrle-reset')) return;
      toggle();
    });

    // Optional: keep button working too, but stop double toggle
    if (hit){
      hit.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        toggle();
      });
    }

    if (resetBtn){
      resetBtn.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        reset();
      });
    }

    // Hover sync
    layers.forEach(el => {
      el.addEventListener('mouseenter', () => { if (open) setActive(Number(el.dataset.idx)); });
      el.addEventListener('mouseleave', () => { if (open) clearActive(); });
    });
    items.forEach(it => {
      it.addEventListener('mouseenter', () => { if (open) setActive(Number(it.dataset.idx)); });
      it.addEventListener('mouseleave', () => { if (open) clearActive(); });
      it.addEventListener('click', () => { if (open) setActive(Number(it.dataset.idx)); });
    });
    callouts.forEach(co => {
      co.addEventListener('mouseenter', () => { if (open) setActive(Number(co.dataset.idx)); });
      co.addEventListener('mouseleave', () => { if (open) clearActive(); });
      co.addEventListener('click', () => { if (open) setActive(Number(co.dataset.idx)); });
    });

    open = startOpen;
    applyTransforms();

    let t=null;
    window.addEventListener('resize', () => {
      clearTimeout(t);
      t=setTimeout(applyTransforms, 120);
    });
  }

  function initAll(){
    document.querySelectorAll('[data-scrle]').forEach(initExploder);
  }

  if (document.readyState === 'loading'){
    document.addEventListener('DOMContentLoaded', initAll);
  } else {
    initAll();
  }
})();
