<?php
/**
 * Plugin Name: Anion Tabs Widget for Elementor
 * Description: Widget de Elementor para Tabs con miniatura + contenido (editable desde el panel del widget). Compatible con Elementor 3.x y PHP 7.4+.
 * Version: 1.0.3
 * Author: SCR Business
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'ANION_EL_TABS_VERSION', '1.0.3' );
define( 'ANION_EL_TABS_PATH', plugin_dir_path( __FILE__ ) );

function anion_el_tabs_is_elementor_active() {
	return did_action( 'elementor/loaded' );
}

add_action( 'admin_notices', function() {
	if ( ! anion_el_tabs_is_elementor_active() && current_user_can( 'activate_plugins' ) ) {
		echo '<div class="notice notice-warning"><p><strong>Anion Tabs Widget</strong>: Este plugin requiere Elementor activo.</p></div>';
	}
} );

function anion_el_tabs_register_assets() {
	if ( ! wp_style_is( 'anion-el-tabs', 'registered' ) ) {
		wp_register_style( 'anion-el-tabs', false, array(), ANION_EL_TABS_VERSION );
	}
	if ( ! wp_script_is( 'anion-el-tabs', 'registered' ) ) {
		wp_register_script( 'anion-el-tabs', false, array(), ANION_EL_TABS_VERSION, true );
	}
}
add_action( 'wp_enqueue_scripts', 'anion_el_tabs_register_assets' );
add_action( 'elementor/editor/before_enqueue_scripts', 'anion_el_tabs_register_assets' );

function anion_el_tabs_enqueue_assets_once() {
	static $done = false;

	anion_el_tabs_register_assets();
	wp_enqueue_style( 'anion-el-tabs' );
	wp_enqueue_script( 'anion-el-tabs' );

	if ( $done ) { return; }
	$done = true;

	$css = <<<CSS
.anion-tabs{max-width:1100px;margin:0 auto}

/* NAV: tabs ocupan todo el ancho */
.anion-tabs__nav{display:flex;gap:18px;align-items:flex-end;justify-content:stretch;flex-wrap:wrap;margin-bottom:18px;width:100%}
.anion-tabs__btn{border:0;background:transparent;padding:0;cursor:pointer;text-align:center;flex:1 1 220px;min-width:180px}
.anion-tabs__thumb{width:160px;height:110px;object-fit:contain;display:block;margin:0 auto 8px;filter:drop-shadow(0 8px 14px rgba(0,0,0,.08))}
.anion-tabs__label{display:block;font-weight:700;font-size:15px;line-height:1.2;color:var(--anion-tab-color,#e91e63)}
.anion-tabs__btn[aria-selected="true"] .anion-tabs__label{color:var(--anion-tab-color-active,#c2185b);text-decoration:underline}
.anion-tabs__btn:hover .anion-tabs__label,
.anion-tabs__btn:focus .anion-tabs__label{color:var(--anion-tab-color-active,#c2185b)}
.anion-tabs__btn:focus{outline:none}

/* PANEL: transparente como solicitaste */
.anion-tabs__panel{display:none;background:transparent;border-radius:14px;padding:18px 0 12px;box-shadow:none}
.anion-tabs__panel.is-active{display:block}

/* opcional: si quieres una "tarjeta" interna, usamos un wrapper */
.anion-tabs__card{background:var(--anion-card-bg, rgba(255,255,255,.70));border-radius:14px;padding:18px 18px 12px;box-shadow:var(--anion-card-shadow, 0 10px 30px rgba(0,0,0,.06));backdrop-filter:saturate(120%) blur(2px)}

.anion-tabs__grid{display:grid;grid-template-columns:1.25fr .75fr;gap:22px;align-items:start}
.anion-tabs__title{margin:0 0 10px;font-size:20px;font-weight:800;color:var(--anion-title-color,#7cb342)}
.anion-tabs__desc{margin:0 0 10px;color:#444;line-height:1.55}
.anion-tabs__subtitle{margin:14px 0 10px;font-weight:800;color:var(--anion-subtitle-color,#e53935)}

/* Lista con flecha (no bullet) */
.anion-tabs__list{margin:0;padding:0;list-style:none;display:grid;grid-template-columns:1fr 1fr;gap:10px 22px}
.anion-tabs__list li{margin:0;color:#333;position:relative;padding-left:18px;line-height:1.35}
.anion-tabs__list li:before{content:"➜";position:absolute;left:0;top:0;color:var(--anion-arrow-color,#7cb342);font-size:14px;line-height:1.35}

.anion-tabs__right img{width:100%;max-width:340px;height:auto;object-fit:contain;display:block;margin-left:auto;filter:drop-shadow(0 10px 20px rgba(0,0,0,.10))}

@media (max-width: 820px){
  .anion-tabs__grid{grid-template-columns:1fr}
  .anion-tabs__right img{margin:12px auto 0;max-width:320px}
  .anion-tabs__list{grid-template-columns:1fr}
  .anion-tabs__nav{gap:14px}
  .anion-tabs__thumb{width:150px;height:105px}
  .anion-tabs__btn{flex:1 1 45%}
}
CSS;

	$js = <<<JS
(function(){
  function initTabs(root){
    var btns = root.querySelectorAll('.anion-tabs__btn');
    var panels = root.querySelectorAll('.anion-tabs__panel');
    if(!btns.length || !panels.length) return;

    function activate(id){
      btns.forEach(function(b){
        var on = b.getAttribute('aria-controls') === id;
        b.setAttribute('aria-selected', on ? 'true' : 'false');
        b.tabIndex = on ? 0 : -1;
      });
      panels.forEach(function(p){
        p.classList.toggle('is-active', p.id === id);
      });
    }

    btns.forEach(function(btn){
      btn.addEventListener('click', function(){ activate(btn.getAttribute('aria-controls')); });
      btn.addEventListener('keydown', function(e){
        var keys = ['ArrowLeft','ArrowRight','Home','End'];
        if(keys.indexOf(e.key) === -1) return;
        e.preventDefault();
        var list = Array.prototype.slice.call(btns);
        var i = list.indexOf(btn);
        var next = i;
        if(e.key==='ArrowLeft') next = (i-1+list.length)%list.length;
        if(e.key==='ArrowRight') next = (i+1)%list.length;
        if(e.key==='Home') next = 0;
        if(e.key==='End') next = list.length-1;
        list[next].focus();
        activate(list[next].getAttribute('aria-controls'));
      });
    });

    activate(btns[0].getAttribute('aria-controls'));
  }

  function boot(scope){
    (scope || document).querySelectorAll('.anion-tabs').forEach(function(el){ initTabs(el); });
  }

  if(document.readyState === 'loading') document.addEventListener('DOMContentLoaded', function(){ boot(document); });
  else boot(document);

  document.addEventListener('elementor/frontend/init', function(){
    try{
      if(window.elementorFrontend && window.elementorFrontend.hooks){
        window.elementorFrontend.hooks.addAction('frontend/element_ready/global', function(scope){
          boot(scope && scope[0] ? scope[0] : document);
        });
      }
    }catch(e){}
    boot(document);
  });
})();
JS;

	wp_add_inline_style( 'anion-el-tabs', $css );
	wp_add_inline_script( 'anion-el-tabs', $js );
}

function anion_el_tabs_register_widget($widgets_manager = null) {
	if ( ! anion_el_tabs_is_elementor_active() ) return;
	if ( ! class_exists( '\\Elementor\\Widget_Base' ) ) return;

	require_once ANION_EL_TABS_PATH . 'widget-anion-tabs.php';

	if ( is_object( $widgets_manager ) && method_exists( $widgets_manager, 'register' ) ) {
		$widgets_manager->register( new \Anion_Elementor_Tabs_Widget() );
	} else {
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Anion_Elementor_Tabs_Widget() );
	}
}

add_action( 'elementor/widgets/register', 'anion_el_tabs_register_widget' );
add_action( 'elementor/widgets/widgets_registered', function() {
	anion_el_tabs_register_widget( \Elementor\Plugin::instance()->widgets_manager );
} );
